package org.moddingx.libx.crafting;

import com.google.gson.JsonObject;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.crafting.Ingredient;

import java.util.function.Predicate;

/**
 * An {@link Ingredient} with an amount.
 */
public record IngredientStack(Ingredient ingredient, int count) implements Predicate<ItemStack> {

    public static final IngredientStack EMPTY = new IngredientStack(Ingredient.f_43901_, 0);
    
    public IngredientStack(Ingredient ingredient, int count) {
        if (count <= 0 || ingredient.m_43947_()) {
            this.ingredient = Ingredient.f_43901_;
            this.count = 0;
        } else {
            this.ingredient = ingredient;
            this.count = count;
        }
    }
    
    /**
     * Returns whether the ingredient matches the stack and the count of the stack is greater or equal
     * to the count of the IngredientStack.
     */
    @Override
    public boolean test(ItemStack stack) {
        return stack.m_41613_() >= this.count && this.ingredient.test(stack);
    }
    
    /**
     * Returns whether the count is 0 or {@link Ingredient#isEmpty()} returns true.
     */
    public boolean isEmpty() {
        return this.count == 0 || this.ingredient.m_43947_();
    }

    /**
     * Serialises the IngredientStack to json.
     */
    public JsonObject toJson() {
        JsonObject json = new JsonObject();
        json.add("Ingredient", this.ingredient.m_43942_());
        json.addProperty("Count", this.count);
        return json;
    }

    /**
     * Writes this IngredientStack to a {@link FriendlyByteBuf}.
     */
    public void toNetwork(FriendlyByteBuf buffer) {
        buffer.m_130130_(this.count);
        this.ingredient.m_43923_(buffer);
    }

    /**
     * Deserializes and IngredientStack from json.
     */
    public static IngredientStack fromJson(JsonObject json) {
        Ingredient ingredient = json.has("Ingredient") ? Ingredient.m_43917_(json.get("Ingredient")) : Ingredient.f_43901_;
        int count = json.has("Count") && json.get("Count").isJsonPrimitive() ? json.get("Count").getAsInt() : 1;
        
        return new IngredientStack(ingredient, count);
    }

    /**
     * Reads an IngredientStack from a {@link FriendlyByteBuf}.
     */
    public static IngredientStack fromNetwork(FriendlyByteBuf buffer) {
        int count = buffer.m_130242_();
        Ingredient ingredient = Ingredient.m_43940_(buffer);
        return new IngredientStack(ingredient, count);
    }
}
