package org.moddingx.libx.creativetab;

import net.minecraft.core.registries.Registries;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.flag.FeatureFlagSet;
import net.minecraft.world.item.*;
import net.minecraftforge.registries.ForgeRegistries;
import net.minecraftforge.registries.ForgeRegistry;
import net.minecraftforge.registries.RegisterEvent;
import org.moddingx.libx.impl.ModInternal;
import org.moddingx.libx.mod.ModX;

import java.util.Comparator;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;

/**
 * Class to register and populate a creative tab. You should subclass it and create an instance in your
 * mods' constructor.
 */
public abstract class CreativeTabX {
    
    // Use numeric ids for ordering as they should represent the order in which items were registered. 
    @SuppressWarnings("UnstableApiUsage")
    private static final Comparator<Item> REGISTRY_ORDER = Comparator.comparing(item -> ForgeRegistries.ITEMS instanceof ForgeRegistry<Item> reg ? reg.getID(item) : Integer.MAX_VALUE - 1);
    
    protected final ModX mod;
    protected final ResourceLocation id;
    private CreativeModeTab tab;
    
    protected CreativeTabX(ModX mod) {
        this(mod, "tab");
    }
    
    protected CreativeTabX(ModX mod, String name) {
        this.mod = mod;
        this.id = mod.resource(name);
        ModInternal.get(mod).modEventBus().addListener(this::registerCreativeTab);
    }

    /**
     * Gets the creative tab.
     * 
     * @throws IllegalArgumentException if the tab has not yet been created.
     */
    public CreativeModeTab tab() {
        if (this.tab == null) {
            throw new IllegalArgumentException("Creative tab " + this.id + " has not yet been created.");
        }
        return this.tab;
    }

    /**
     * Sets some basic creative tab information. This is not for tab contents. The default implementation just
     * sets the tab title.
     */
    protected void buildTab(CreativeModeTab.Builder builder) {
        builder.m_257941_(Component.m_237115_("itemGroup." + this.id.m_135827_() + ("tab".equals(this.id.m_135815_()) ? "" : "." + this.id.m_135815_())));
    }

    /**
     * Adds the items to the tab.
     */
    protected abstract void addItems(TabContext ctx);

    /**
     * Adds all items from the current mod into the tab.
     */
    protected void addModItems(TabContext ctx) {
        this.addModItems(ctx, REGISTRY_ORDER);
    }

    /**
     * Adds all items from the current mod into the tab using a custom order.
     */
    protected void addModItems(TabContext ctx, Comparator<Item> order) {
        this.addModItems(ctx, order, item -> ctx.context().f_268429_() || !(item instanceof GameMasterBlockItem));
    }

    /**
     * Adds all items from the current mod, that match a predicate into the tab.
     */
    protected void addModItems(TabContext ctx, Predicate<Item> items) {
        this.addModItems(ctx, REGISTRY_ORDER, items);
    }
    
    /**
     * Adds all items from the current mod, that match a predicate into the tab using a custom order.
     */
    protected void addModItems(TabContext ctx, Comparator<Item> order, Predicate<Item> items) {
        this.addModItemStacks(ctx, order, item -> items.test(item) ? this.itemStream(item) : Stream.empty());
    }
    
    /**
     * Generates a stream of {@link ItemStack item stacks} for each item from the current mod and
     * adds the stacks to the tab.
     */
    protected void addModItemStacks(TabContext ctx, Function<Item, Stream<ItemStack>> stacks) {
        this.addModItemStacks(ctx, REGISTRY_ORDER, stacks);
    }
    
    /**
     * Generates a stream of {@link ItemStack item stacks} for each item from the current mod and
     * adds the stacks to the tab using a custom order.
     */
    protected void addModItemStacks(TabContext ctx, Comparator<Item> order, Function<Item, Stream<ItemStack>> stacks) {
        ForgeRegistries.ITEMS.getEntries().stream()
                .filter(entry -> this.mod.modid.equals(entry.getKey().m_135782_().m_135827_()))
                .map(Map.Entry::getValue)
                .filter(item -> item.m_245183_().m_247715_(ctx.features()))
                .sorted(order)
                .flatMap(stacks)
                .forEach(stack -> ctx.output().m_246342_(stack));
    }
    
    private void registerCreativeTab(RegisterEvent event) {
        event.register(Registries.f_279569_, reg -> {
            CreativeModeTab.Builder builder = CreativeModeTab.builder();
            this.buildTab(builder);
            builder.m_257501_((context, output) -> this.addItems(new TabContext(context, context.f_268709_(), output)));
            this.tab = builder.m_257652_();
            reg.register(this.id, this.tab);
        });
    }
    
    private Stream<ItemStack> itemStream(Item item) {
        if (item instanceof CreativeTabItemProvider provider) {
            return provider.makeCreativeTabStacks();
        } else {
            return Stream.of(new ItemStack(item));
        }
    }
    
    public record TabContext(CreativeModeTab.ItemDisplayParameters context, FeatureFlagSet features, CreativeModeTab.Output output) {}
}
