package org.moddingx.libx.datagen.provider.loot;

import net.minecraft.advancements.critereon.EnchantmentPredicate;
import net.minecraft.advancements.critereon.ItemPredicate;
import net.minecraft.advancements.critereon.MinMaxBounds;
import net.minecraft.advancements.critereon.StatePropertiesPredicate;
import net.minecraft.core.registries.Registries;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.tags.TagKey;
import net.minecraft.util.StringRepresentable;
import net.minecraft.world.item.Item;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.enchantment.Enchantment;
import net.minecraft.world.item.enchantment.Enchantments;
import net.minecraft.world.level.ItemLike;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.block.state.properties.BedPart;
import net.minecraft.world.level.block.state.properties.BlockStateProperties;
import net.minecraft.world.level.block.state.properties.DoubleBlockHalf;
import net.minecraft.world.level.block.state.properties.Property;
import net.minecraft.world.level.storage.loot.BuiltInLootTables;
import net.minecraft.world.level.storage.loot.LootPool;
import net.minecraft.world.level.storage.loot.LootTable;
import net.minecraft.world.level.storage.loot.entries.LootItem;
import net.minecraft.world.level.storage.loot.entries.LootPoolEntryContainer;
import net.minecraft.world.level.storage.loot.entries.LootPoolSingletonContainer;
import net.minecraft.world.level.storage.loot.functions.ApplyBonusCount;
import net.minecraft.world.level.storage.loot.functions.CopyBlockState;
import net.minecraft.world.level.storage.loot.functions.CopyNbtFunction;
import net.minecraft.world.level.storage.loot.parameters.LootContextParamSets;
import net.minecraft.world.level.storage.loot.predicates.*;
import net.minecraft.world.level.storage.loot.providers.nbt.ContextNbtProvider;
import net.minecraft.world.level.storage.loot.providers.number.ConstantValue;
import org.moddingx.libx.datagen.DatagenContext;
import org.moddingx.libx.datagen.loot.LootBuilders;
import org.moddingx.libx.datagen.provider.loot.entry.GenericLootModifier;
import org.moddingx.libx.datagen.provider.loot.entry.LootFactory;
import org.moddingx.libx.datagen.provider.loot.entry.LootModifier;
import org.moddingx.libx.datagen.provider.loot.entry.SimpleLootFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.List;

public abstract class BlockLootProviderBase extends LootProviderBase<Block> {

    protected BlockLootProviderBase(DatagenContext ctx) {
        super(ctx, "blocks", LootContextParamSets.f_81421_, Registries.f_256747_);
    }
    
    @Nullable
    @Override
    protected LootTable.Builder defaultBehavior(Block block) {
        if (block.m_49965_().m_61056_().stream().anyMatch(this::needsLootTable)) {
            LootPoolEntryContainer.Builder<?> entry = LootItem.m_79579_(block);
            LootPool.Builder pool = LootPool.m_79043_().m_165133_(ConstantValue.m_165692_(1)).m_79076_(entry)
                    .m_79080_(ExplosionCondition.m_81661_());
            if (block.m_49966_().m_61138_(BlockStateProperties.f_61401_)) {
                pool = pool.m_79080_(LootItemBlockStatePropertyCondition.m_81769_(block).m_81784_(
                        StatePropertiesPredicate.Builder.m_67693_().m_67697_(BlockStateProperties.f_61401_, DoubleBlockHalf.LOWER)
                ));
            }
            if (block.m_49966_().m_61138_(BlockStateProperties.f_61391_)) {
                pool = pool.m_79080_(LootItemBlockStatePropertyCondition.m_81769_(block).m_81784_(
                        StatePropertiesPredicate.Builder.m_67693_().m_67697_(BlockStateProperties.f_61391_, BedPart.HEAD)
                ));
            }
            return LootTable.m_79147_().m_79161_(pool);
        } else {
            return null;
        }
    }

    /**
     * Returns whether this block state needs a loot table. If all block states of a block don't
     * need a loot table, defaultBehavior will return null for that block. Can be overridden to
     * alter the behaviour.
     */
    protected boolean needsLootTable(BlockState state) {
        return !state.m_60795_() && state.m_60819_().m_76188_().m_60734_() != state.m_60734_()
                && !BuiltInLootTables.f_78712_.equals(state.m_60734_().m_60589_());
    }

    @Override
    public void generateBaseTable(Block block, LootPoolEntryContainer.Builder<?> entry) {
        LootPool.Builder pool = LootPool.m_79043_()
                .m_165133_(ConstantValue.m_165692_(1)).m_79076_(entry)
                .m_79080_(ExplosionCondition.m_81661_());
        this.customLootTable(block, LootTable.m_79147_().m_79161_(pool));
    }

    @Override
    protected SimpleLootFactory<Block> element() {
        return LootItem::m_79579_;
    }

    @Override
    public void drops(Block block, ItemStack... drops) {
        this.drops(block, true, drops);
    }

    @Override
    public void drops(Block block, List<LootFactory<Block>> loot) {
        this.drops(block, true, loot);
    }

    public void drops(Block block, boolean silkTouch, ItemStack... drops) {
        this.drops(block, silkTouch ? this.silk(this.identity()) : this.noSilk(), drops);
    }
    
    @SafeVarargs
    public final void drops(Block block, boolean silkTouch, LootFactory<Block>... drops) {
        this.drops(block, silkTouch, Arrays.stream(drops).toList());
    }
    
    public void drops(Block block, boolean silkTouch, List<LootFactory<Block>> drops) {
        this.drops(block, silkTouch ? this.silk(this.identity()) : this.noSilk(), drops);
    }
    
    public void drops(Block block, SilkModifier silkTouch, ItemStack... drops) {
        this.drops(block, silkTouch, Arrays.stream(drops).<LootFactory<Block>>map(this::stack).toList());
    }
    
    @SafeVarargs
    public final void drops(Block block, SilkModifier silkTouch, LootFactory<Block>... drops) {
        this.drops(block, silkTouch, Arrays.stream(drops).toList());
    }
    
    public void drops(Block block, SilkModifier silkTouch, List<LootFactory<Block>> drops) {
        LootPoolEntryContainer.Builder<?> entry = this.combine(drops).build(block);
        if (silkTouch.modifier != null) {
            LootPoolEntryContainer.Builder<?> silkBuilder = silkTouch.modifier.apply(block, this.element().build(block)).m_79080_(this.silkCondition());
            entry = LootBuilders.alternative(List.of(silkBuilder, entry));
        }
        this.generateBaseTable(block, entry);
    }
    
    /**
     * Turns a generic loot modifier into a silk modifier. This exists to reduce ambiguity.
     * A silk modifier does not extend {@link GenericLootModifier} for this reason.
     */
    public SilkModifier silk(GenericLootModifier<Block> modifier) {
        return new SilkModifier(modifier);
    }
    
    /**
     * Gets a new silk modifier that means: No special silk touch behaviour.
     */
    public SilkModifier noSilk() {
        return new SilkModifier(null);
    }
    
    /**
     * A loot modifier to apply fortune based on the formula used for ores.
     */
    public LootModifier<Block> fortuneOres() {
        return this.modifier((block, entry) -> entry.m_79078_(ApplyBonusCount.m_79915_(Enchantments.f_44987_)));
    }
    
    /**
     * A loot modifier to apply fortune based on a uniform formula.
     */
    public LootModifier<Block> fortuneUniform() {
        return this.fortuneUniform(1);
    }
    
    /**
     * A loot modifier to apply fortune based on a uniform formula.
     */
    public LootModifier<Block> fortuneUniform(int multiplier) {
        return this.modifier((block, entry) -> entry.m_79078_(ApplyBonusCount.m_79921_(Enchantments.f_44987_, multiplier)));
    }

    /**
     * A loot modifier to apply fortune based on a binomial formula.
     */
    public LootModifier<Block> fortuneBinomial(float probability) {
        return this.fortuneBinomial(probability, 0);
    }
    
    /**
     * A loot modifier to apply fortune based on a binomial formula.
     */
    public LootModifier<Block> fortuneBinomial(float probability, int bonus) {
        return this.modifier((block, entry) -> entry.m_79078_(ApplyBonusCount.m_79917_(Enchantments.f_44987_, probability, bonus)));
    }
    
    
    /**
     * A condition that is random with a chance and optionally different chances for
     * different fortune levels. Chances for different levels are computed automatically.
     */
    public LootItemCondition.Builder randomFortune(float baseChance) {
        return this.randomFortune(baseChance, baseChance * (10/9f), baseChance * 1.25f, baseChance * (5/3f), baseChance * 5);
    }
    
    /**
     * A condition that is random with a chance and optionally different chances for
     * different fortune levels.
     * 
     * @param baseChance The chance without fortune.
     * @param levelChances the chances with fortune.
     */
    public LootItemCondition.Builder randomFortune(float baseChance, float... levelChances) {
        float[] chances = new float[levelChances.length + 1];
        chances[0] = baseChance;
        System.arraycopy(levelChances, 0, chances, 1, levelChances.length);
        return BonusLevelTableCondition.m_81517_(Enchantments.f_44987_, chances);
    }

    /**
     * Gets a loot condition builder for a match tool condition.
     */
    public MatchToolBuilder matchTool(ItemLike item) {
        return new MatchToolBuilder(ItemPredicate.Builder.m_45068_().m_151445_(item));
    }

    /**
     * Gets a loot condition builder for a match tool condition.
     */
    public MatchToolBuilder matchTool(TagKey<Item> item) {
        return new MatchToolBuilder(ItemPredicate.Builder.m_45068_().m_204145_(item));
    }

    /**
     * Gets a loot modifier builder for a match state condition.
     */
    public MatchStateBuilder matchState() {
        return new MatchStateBuilder(StatePropertiesPredicate.Builder.m_67693_());
    }

    /**
     * Gets a loot condition for silk touch tools.
     */
    public LootItemCondition.Builder silkCondition() {
        ItemPredicate.Builder predicate = ItemPredicate.Builder.m_45068_()
                .m_45071_(new EnchantmentPredicate(Enchantments.f_44985_, MinMaxBounds.Ints.m_55386_(1)));
        return MatchTool.m_81997_(predicate);
    }

    /**
     * Creates a loot modifier that copies NBT-Data from a block entity into the dropped item.
     *
     * @param tags The toplevel tags of the block entity to be copied.
     */
    public LootModifier<Block> copyNBT(String... tags) {
        return this.modifier((block, entry) -> {
            CopyNbtFunction.Builder func = CopyNbtFunction.m_165180_(ContextNbtProvider.f_165562_);
            for (String tag : tags) {
                func = func.m_80279_(tag, "BlockEntityTag." + tag);
            }
            return entry.m_79078_(func);
        });
    }

    /**
     * Creates a loot modifier that copies properties from a block state into the dropped item.
     *
     * @param properties The properties of the block state to be copied.
     */
    public LootModifier<Block> copyProperties(Property<?>... properties) {
        return this.modifier((block, entry) -> {
            CopyBlockState.Builder func = CopyBlockState.m_80062_(block);
            for (Property<?> property : properties) {
                func = func.m_80084_(property);
            }
            return entry.m_79078_(func);
        });
    }
    
     /**
     * A class used in the drops method to reduce ambiguity and make the code more readable. Just call
     * the {@link #silk(GenericLootModifier)} method wih a generic loot modifier to get a silk modifier.
     * A silk modifier does not extend {@link GenericLootModifier} for this reason.
     */
    public static class SilkModifier {
        
        @Nullable
        public final GenericLootModifier<Block> modifier;

        private SilkModifier(@Nullable GenericLootModifier<Block> modifier) {
            this.modifier = modifier;
        }
    }

    /**
     * This serves as a builder for a loot condition and a builder for a match tool predicate
     * in one.
     */
    public static class MatchToolBuilder implements LootItemCondition.Builder {
        
        private final ItemPredicate.Builder builder;

        private MatchToolBuilder(ItemPredicate.Builder builder) {
            this.builder = builder;
        }

        @Nonnull
        @Override
        public LootItemCondition m_6409_() {
            return MatchTool.m_81997_(this.builder).m_6409_();
        }

        /**
         * Adds a required enchantment to this builder.
         */
        public MatchToolBuilder enchantment(Enchantment enchantment) {
            return this.enchantment(enchantment, MinMaxBounds.Ints.m_55386_(1));
        }
        
        /**
         * Adds a required enchantment to this builder.
         * 
         * @param minLevel The minimum level of the enchantment that must be present.
         */
        public MatchToolBuilder enchantment(Enchantment enchantment, int minLevel) {
            return this.enchantment(enchantment, MinMaxBounds.Ints.m_55386_(minLevel));
        }
        
        /**
         * Adds a required enchantment to this builder.
         * 
         * @param level The exact level of the enchantment that must be present.
         */
        public MatchToolBuilder enchantmentExact(Enchantment enchantment, int level) {
            return this.enchantment(enchantment, MinMaxBounds.Ints.m_55371_(level));
        }
        
        private MatchToolBuilder enchantment(Enchantment enchantment, MinMaxBounds.Ints bounds) {
            this.builder.m_45071_(new EnchantmentPredicate(enchantment, bounds));
            return this;
        }

        /**
         * Adds required NBT data to this builder.
         */
        public MatchToolBuilder nbt(CompoundTag nbt) {
            this.builder.m_45075_(nbt);
            return this;
        }
    }

    /**
     * This serves as a builder for a loot modifier and a builder for a block state predicate
     * in one.
     */
    public class MatchStateBuilder implements GenericLootModifier<Block> {
        
        private final StatePropertiesPredicate.Builder builder;

        private MatchStateBuilder(StatePropertiesPredicate.Builder builder) {
            this.builder = builder;
        }

        @Override
        public LootPoolEntryContainer.Builder<?> apply(Block item, LootPoolSingletonContainer.Builder<?> entry) {
            return entry.m_79080_(LootItemBlockStatePropertyCondition.m_81769_(item).m_81784_(this.builder));
        }

        @Override
        public SimpleLootFactory<Block> element() {
            return BlockLootProviderBase.this.element();
        }

         public MatchStateBuilder hasProperty(Property<?> property, String value) {
             this.builder.m_67700_(property, value);
             return this;
          }

          public MatchStateBuilder hasProperty(Property<Integer> property, int value) {
              this.builder.m_67694_(property, value);
              return this;
          }

          public MatchStateBuilder hasProperty(Property<Boolean> property, boolean value) {
              this.builder.m_67703_(property, value);
              return this;
          }

          public <T extends Comparable<T> & StringRepresentable> MatchStateBuilder hasProperty(Property<T> property, T value) {
              this.builder.m_67697_(property, value);
              return this;
          }
    }
}
