package org.moddingx.libx.datagen.provider.loot;

import com.google.common.collect.Multimap;
import net.minecraft.core.Registry;
import net.minecraft.data.CachedOutput;
import net.minecraft.data.DataProvider;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.packs.PackType;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.level.ItemLike;
import net.minecraft.world.level.storage.loot.*;
import net.minecraft.world.level.storage.loot.entries.*;
import net.minecraft.world.level.storage.loot.functions.LootItemConditionalFunction;
import net.minecraft.world.level.storage.loot.functions.SetItemCountFunction;
import net.minecraft.world.level.storage.loot.parameters.LootContextParamSet;
import net.minecraft.world.level.storage.loot.parameters.LootContextParamSets;
import net.minecraft.world.level.storage.loot.predicates.*;
import net.minecraft.world.level.storage.loot.providers.number.BinomialDistributionGenerator;
import net.minecraft.world.level.storage.loot.providers.number.ConstantValue;
import net.minecraft.world.level.storage.loot.providers.number.UniformGenerator;
import net.minecraftforge.common.data.ExistingFileHelper;
import org.moddingx.libx.LibX;
import org.moddingx.libx.datagen.DatagenContext;
import org.moddingx.libx.datagen.PackTarget;
import org.moddingx.libx.datagen.loot.LootBuilders;
import org.moddingx.libx.datagen.provider.loot.entry.GenericLootModifier;
import org.moddingx.libx.datagen.provider.loot.entry.LootFactory;
import org.moddingx.libx.datagen.provider.loot.entry.LootModifier;
import org.moddingx.libx.datagen.provider.loot.entry.SimpleLootFactory;
import org.moddingx.libx.impl.datagen.loot.LootData;
import org.moddingx.libx.mod.ModX;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public abstract class LootProviderBase<T> implements DataProvider {

    private static final ExistingFileHelper.IResourceType LOOT_TYPE = new ExistingFileHelper.ResourceType(PackType.SERVER_DATA, ".json", "loot_tables");

    protected final ModX mod;
    protected final PackTarget packTarget;
    protected final ExistingFileHelper fileHelper;
    protected final String folder;
    protected final LootContextParamSet params;
    protected final Supplier<Stream<Map.Entry<ResourceLocation, T>>> modElements;
    protected final Function<T, ResourceLocation> idResolver;

    private final Set<T> ignored = new HashSet<>();
    private final Map<T, Function<T, LootTable.Builder>> functionMap = new HashMap<>();

    protected LootProviderBase(DatagenContext ctx, String folder, LootContextParamSet params, ResourceKey<? extends Registry<T>> registryKey) {
        this(ctx, folder, params,
                () -> ctx.registries().registry(registryKey).m_6579_().stream()
                        .sorted(Map.Entry.comparingByKey(Comparator.comparing(ResourceKey::m_135782_)))
                        .map(entry -> Map.entry(entry.getKey().m_135782_(), entry.getValue())),
                id -> ctx.registries().registry(registryKey).m_7981_(id)
        );
    }
    
    protected LootProviderBase(DatagenContext ctx, String folder, LootContextParamSet params, Supplier<Stream<Map.Entry<ResourceLocation, T>>> modElements, Function<T, ResourceLocation> allElementIds) {
        this.mod = ctx.mod();
        this.packTarget = ctx.target();
        this.fileHelper = ctx.fileHelper();
        this.folder = folder;
        this.params = params;
        this.modElements = modElements;
        this.idResolver = value -> {
            ResourceLocation id = allElementIds.apply(value);
            if (id == null) throw new IllegalStateException("Unregistered value: " + value);
            return id;
        };
    }
    
    protected LootProviderBase(DatagenContext ctx, String folder, LootContextParamSet params, Function<T, ResourceLocation> elementIds) {
        this.mod = ctx.mod();
        this.packTarget = ctx.target();
        this.fileHelper = ctx.fileHelper();
        this.folder = folder;
        this.params = params;
        this.modElements = () -> this.functionMap.keySet().stream().map(element -> Map.entry(elementIds.apply(element), element));
        this.idResolver = value -> {
            ResourceLocation id = elementIds.apply(value);
            if (id == null) throw new IllegalStateException("Unregistered value: " + value);
            return id;
        };
    }

    protected abstract void setup();

    /**
     * The given item will not be processed by this provider. Useful when you want to create the loot table manually.
     */
    protected void customLootTable(T item) {
        this.ignored.add(item);
    }

    /**
     * The given item will get the given loot table.
     */
    protected void customLootTable(T item, LootTable.Builder loot) {
        this.functionMap.put(item, b -> loot);
    }

    /**
     * The given item will get the given loot table function.
     */
    protected void customLootTable(T item, Function<T, LootTable.Builder> loot) {
        this.functionMap.put(item, loot);
    }

    /**
     * Gets the element factory. The element factory is a loot factory that assigns
     * each element a loot entry. The default just assigns an empty entry.
     */
    protected SimpleLootFactory<T> element() {
        return SimpleLootFactory.from(EmptyLootItem.m_79533_());
    }
    
    /**
     * Creates a default loot table for the given item. Can be overridden to alter
     * default behaviour. Should return null if no loot table should be generated.
     */
    @Nullable
    protected abstract LootTable.Builder defaultBehavior(T item);

    @Nonnull
    @Override
    public String m_6055_() {
        ResourceLocation key = LootContextParamSets.m_81426_(this.params);
        String name = key == null ? "generic" : ("minecraft".equals(key.m_135827_()) ? key.m_135815_() : key.toString());
        return this.mod.modid + " " + name + " loot tables";
    }
    
    @Nonnull
    @Override
    public CompletableFuture<?> m_213708_(@Nonnull CachedOutput cache) {
        this.setup();

        Map<ResourceLocation, LootTable> tables = this.modElements.get()
                .filter(entry -> this.mod.modid.equals(entry.getKey().m_135827_()))
                .filter(entry -> !this.ignored.contains(entry.getValue()))
                .flatMap(this::resolve)
                .map(entry -> Map.entry(new ResourceLocation(entry.getKey().m_135827_(), this.folder + "/" + entry.getKey().m_135815_()), entry.getValue()))
                .collect(Collectors.toUnmodifiableMap(Map.Entry::getKey, Map.Entry::getValue));

        ValidationContext validationContext = new ValidationContext(this.params, new LootDataResolver() {
            
            @Nullable
            @Override
            @SuppressWarnings("unchecked")
            public <A> A m_278667_(@Nonnull LootDataId<A> id) {
                if (id.f_278383_() != LootDataType.f_278413_) return null;
                if (tables.containsKey(id.f_278500_())) return (A) tables.get(id.f_278500_());
                if (LootProviderBase.this.fileHelper.exists(id.f_278500_(), LOOT_TYPE)) return (A) LootTable.m_79147_().m_79167_();
                return null;
            }
        });
        
        for (Map.Entry<ResourceLocation, LootTable> entry : tables.entrySet()) {
            entry.getValue().m_79136_(validationContext.m_278632_("{" + entry.getKey() + "}", new LootDataId<>(LootDataType.f_278413_, entry.getKey())));
        }
        
        for (Map.Entry<ResourceLocation, LootTable> entry : tables.entrySet()) {
            this.fileHelper.trackGenerated(entry.getKey(), LOOT_TYPE);
        }
        
        Multimap<String, String> multimap = validationContext.m_79352_();
        if (!multimap.isEmpty()) {
            multimap.forEach((where, what) -> LibX.logger.warn("LootTable validation problem in " + where + ": " + what)); 
            throw new IllegalStateException("There were problems validating the loot tables.");
        }
        
        return CompletableFuture.allOf(tables.entrySet().stream().map(entry -> {
            Path path = this.packTarget.path(PackType.SERVER_DATA).resolve(entry.getKey().m_135827_()).resolve("loot_tables").resolve(entry.getKey().m_135815_() + ".json");
            return DataProvider.m_253162_(cache, LootDataType.f_278413_.m_278857_().toJsonTree(entry.getValue()), path);
        }).toArray(CompletableFuture[]::new));
    }
    
    private Stream<Map.Entry<ResourceLocation, LootTable>> resolve(Map.Entry<ResourceLocation, T> entry) {
        Function<T, LootTable.Builder> loot;
        if (this.functionMap.containsKey(entry.getValue())) {
            loot = this.functionMap.get(entry.getValue());
        } else {
            LootTable.Builder builder = this.defaultBehavior(entry.getValue());
            loot = builder == null ? null : b -> builder;
        }
        return loot == null ? Stream.empty() : Stream.of(Map.entry(entry.getKey(), loot.apply(entry.getValue()).m_79165_(this.params).m_79167_()));
    }

    protected final LootModifier<T> modifier(BiFunction<T, LootPoolSingletonContainer.Builder<?>, LootPoolSingletonContainer.Builder<?>> function) {
        return LootModifier.of(this.element(), function);
    }
    
    protected final GenericLootModifier<T> genericModifier(BiFunction<T, LootPoolSingletonContainer.Builder<?>, LootPoolEntryContainer.Builder<?>> function) {
        return GenericLootModifier.of(this.element(), function);
    }
    
    protected final LootModifier<T> identity() {
        return LootModifier.identity(this.element());
    }

    /**
     * Method to add a custom loot table for an item.
     */
    public void drops(T item, ItemStack... drops) {
        this.drops(item, Arrays.stream(drops).<LootFactory<T>>map(this::stack).toList());
    }
    
    /**
     * Method to add a custom loot table for an item.
     */
    @SafeVarargs
    public final void drops(T item, LootFactory<T>... loot) {
        this.drops(item, Arrays.stream(loot).toList());
    }
    
    /**
     * Method to add a custom loot table for an item.
     */
    public void drops(T item, List<LootFactory<T>> loot) {
        this.generateBaseTable(item, this.combine(loot).build(item));
    }
    
    /**
     * Generate the base loot table.
     */
    public void generateBaseTable(T item, LootPoolEntryContainer.Builder<?> entry) {
        LootPool.Builder pool = LootPool.m_79043_()
                .m_165133_(ConstantValue.m_165692_(1)).m_79076_(entry);
        this.customLootTable(item, LootTable.m_79147_().m_79161_(pool));
    }
    
    /**
     * Turns a singleton loot entry into a simple loot factory.
     */
    public SimpleLootFactory<T> from(LootPoolSingletonContainer.Builder<?> entry) {
        return SimpleLootFactory.from(entry);
    }

    /**
     * Turns a loot entry into a loot factory.
     */
    public LootFactory<T> from(LootPoolEntryContainer.Builder<?> entry) {
        return LootFactory.from(entry);
    }

    /**
     * Turns a loot function into a loot modifier.
     */
    public LootModifier<T> from(LootItemConditionalFunction.Builder<?> function) {
        return this.modifier((item, entry) -> entry.m_79078_(function));
    }

    /**
     * Makes a reference to another loot table in this mod.
     */
    public SimpleLootFactory<T> reference(T value) {
        ResourceLocation elementId = this.idResolver.apply(value);
        return this.reference(new ResourceLocation(elementId.m_135827_(), this.folder + "/" + elementId.m_135815_()));
    }

    /**
     * Makes a reference to another loot table.
     */
    public SimpleLootFactory<T> reference(ResourceLocation lootTable) {
        return SimpleLootFactory.from(LootTableReference.m_79776_(lootTable));
    }
    
    /**
     * A condition that is random with a chance.
     */
    public LootItemCondition.Builder random(float chance) {
        return LootItemRandomChanceCondition.m_81927_(chance);
    }

    /**
     * A loot modifier that sets the count of a stack.
     */
    public LootModifier<T> count(int count) {
        return this.from(SetItemCountFunction.m_165412_(ConstantValue.m_165692_(count)));
    }

    /**
     * A loot modifier that uniformly sets the count of a stack between two values.
     */
    public LootModifier<T> count(int min, int max) {
        if (min == max) {
            return this.from(SetItemCountFunction.m_165412_(ConstantValue.m_165692_(min)));
        } else {
            return this.from(SetItemCountFunction.m_165412_(UniformGenerator.m_165780_(min, max)));
        }
    }

    /**
     * A loot modifier that sets the count of a stack based on a binomial formula.
     */
    public LootModifier<T> countBinomial(float chance, int num) {
        return this.from(SetItemCountFunction.m_165412_(BinomialDistributionGenerator.m_165659_(num, chance)));
    }

    /**
     * Inverts a loot condition
     */
    public LootItemCondition.Builder not(LootItemCondition.Builder condition) {
        return InvertedLootItemCondition.m_81694_(condition);
    }
    
    /**
     * Creates a condition that is met when all of the given conditions are met.
     */
    public LootItemCondition.Builder and(LootItemCondition.Builder... conditions) {
        return AllOfCondition.m_285871_(conditions);
    }

    /**
     * Creates a condition that is met when at least one of the given conditions is met.
     */
    public LootItemCondition.Builder or(LootItemCondition.Builder... conditions) {
        return AnyOfCondition.m_285758_(conditions);
    }

    /**
     * A loot factory for a specific item.
     */
    public SimpleLootFactory<T> stack(ItemLike item) {
        return this.from(LootItem.m_79579_(item));
    }

    /**
     * Tries to create the best possible representation of stack in a loot entry.
     */
    public SimpleLootFactory<T> stack(ItemStack stack) {
        return this.from(LootData.stack(stack));
    }
    
    /**
     * Combines the given loot factories into one. (All loot factories will be applied).
     */
    @SafeVarargs
    public final LootFactory<T> combine(LootFactory<T>... loot) {
        return this.combine(Arrays.stream(loot).toList());
    }
    
    /**
     * Combines the given loot factories into one. (All loot factories will be applied).
     */
    public final LootFactory<T> combine(List<LootFactory<T>> loot) {
        return e -> LootData.combineBy(LootBuilders::all, l -> l.build(e), loot);
    }

    /**
     * Combines the given loot factories into one. (One loot factory will be applied).
     */
    @SafeVarargs
    public final LootFactory<T> random(LootFactory<T>... loot) {
        return this.random(Arrays.stream(loot).toList());
    
    }
    
    /**
     * Combines the given loot factories into one. (One loot factory will be applied).
     */
    public final LootFactory<T> random(List<LootFactory<T>> loot) {
        return e -> LootData.combineBy(LootBuilders::group, l -> l.build(e), loot);
    }

    /**
     * Combines the given loot factories into one. Only the first matching factory is applied.
     */
    @SafeVarargs
    public final LootFactory<T> first(LootFactory<T>... loot) {
        return this.first(Arrays.stream(loot).toList());
    
    }
    
    /**
     * Combines the given loot factories into one. Only the first matching factory is applied.
     */
    public final LootFactory<T> first(List<LootFactory<T>> loot) {
        return e -> LootData.combineBy(LootBuilders::alternative, l -> l.build(e), loot);
    }
    
    /**
     * Combines the given loot factories into one.
     * From all the loot factories until the first one not matching, one is selected.
     */
    @SafeVarargs
    public final LootFactory<T> whileMatch(LootFactory<T>... loot) {
        return this.whileMatch(Arrays.stream(loot).toList());
    }
    
    /**
     * Combines the given loot factories into one.
     * From all the loot factories until the first one not matching, one is selected.
     */
    public final LootFactory<T> whileMatch(List<LootFactory<T>> loot) {
        return e -> LootData.combineBy(LootBuilders::sequence, l -> l.build(e), loot);
    }
}
