package org.moddingx.libx.datagen.provider.patchouli.page;

import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;
import net.minecraft.client.StringSplitter;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.FormattedText;
import net.minecraft.network.chat.MutableComponent;
import net.minecraft.network.chat.Style;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.item.ItemStack;
import net.minecraftforge.registries.ForgeRegistries;
import org.moddingx.libx.impl.datagen.load.DatagenFontLoader;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Some utilities for creating pages in a patchouli book.
 */
public class PageJson {

    /**
     * Gets a {@link JsonElement} that represents the given {@link ItemStack} in the format, patchouli requires it.
     */
    public static JsonElement stack(ItemStack stack) {
        StringBuilder sb = new StringBuilder();
        ResourceLocation id = ForgeRegistries.ITEMS.getKey(stack.m_41720_());
        if (id == null) throw new IllegalStateException("Item not registered: " + stack);
        sb.append(id.m_135827_());
        sb.append(":");
        sb.append(id.m_135815_());
        if (stack.m_41613_() != 1) {
            sb.append("#");
            sb.append(stack.m_41613_());
        }
        if (stack.m_41782_() && !stack.m_41784_().m_128456_()) {
            sb.append(stack.m_41784_());
        }
        return new JsonPrimitive(sb.toString());
    }

    /**
     * Splits the given text onto multiple full text pages.
     */
    public static List<String> splitText(String text) {
        return splitText(text, false);
    }
    
    /**
     * Splits the given text onto multiple full text pages.
     * 
     * @param withInit Whether the first text page should be a little bit smaller, so there is enough space for the
     *                 entry header.
     */
    public static List<String> splitText(String text, boolean withInit) {
        return splitText(text, withInit ? 14 : 16, 16);
    }

    /**
     * Splits the given text onto multiple text pages.
     * 
     * @param skip How many lines, the first page should be shorter that usual.
     */
    public static List<String> splitText(String text, int skip) {
        return splitText(text, Math.max(16 - skip, 1), 16);
    }
    
    /**
     * Splits the given text onto multiple text pages.
     * 
     * @param linesHead The amount of lines on the first page.
     * @param linesTail The amount of lines on all other pages.
     */
    public static List<String> splitText(String text, int linesHead, int linesTail) {
        Component displayText = displayText(text);
        StringSplitter splitter = DatagenFontLoader.getFontMetrics(null);
        List<String> lines = splitter.m_92414_(displayText, Math.round(116 * 1.65f), Style.f_131099_).stream().map(FormattedText::getString).map(String::strip).filter(s -> !s.isEmpty()).toList();
        List<String> pages = new ArrayList<>();
        boolean first = true;
        while (!lines.isEmpty()) {
            pages.add(lines.stream().limit(first ? linesHead : linesTail).collect(Collectors.joining(" ")));
            lines = lines.stream().skip(first ? linesHead : linesTail).toList();
            first = false;
        }
        return List.copyOf(pages);
    }
    
    // Make a component, where formatting codes use a marker for a zero width font recognised by the splitter.
    private static Component displayText(String text) {
        Style zeroWidth = Style.f_131099_.m_131150_(DatagenFontLoader.ZERO_WIDTH_FONT);
        MutableComponent display = Component.m_237119_();
        
        StringBuilder current = new StringBuilder();
        StringBuilder currentFmt = new StringBuilder();
        for (int idx = 0; idx < text.length();) {
            if (text.charAt(idx) == '$' && idx + 1 < text.length() && text.charAt(idx + 1) == '(') {
                if (!current.isEmpty()) {
                    display.m_7220_(Component.m_237113_(current.toString()).m_6270_(Style.f_131099_));
                    current = new StringBuilder();
                }
                int openCodes = 1;
                currentFmt.append("$(");
                idx += 2;
                while (openCodes > 0 && idx < text.length()) {
                    // Handle nested codes
                    if (text.charAt(idx) == '$' && idx + 1 < text.length() && text.charAt(idx + 1) == '(') {
                        currentFmt.append("$(");
                        openCodes += 1;
                        idx += 2;
                    } else if (text.charAt(idx) == ')') {
                        currentFmt.append(")");
                        openCodes -= 1;
                        idx += 1;
                    } else {
                        currentFmt.append(text.charAt(idx));
                        idx += 1;
                    }
                }
            } else {
                if (!currentFmt.isEmpty()) {
                    display.m_7220_(Component.m_237113_(currentFmt.toString()).m_6270_(zeroWidth));
                    currentFmt = new StringBuilder();
                }
                current.append(text.charAt(idx));
                idx += 1;
            }
        }
        if (!current.isEmpty()) {
            display.m_7220_(Component.m_237113_(current.toString()).m_6270_(Style.f_131099_));
        }
        if (!currentFmt.isEmpty()) {
            display.m_7220_(Component.m_237113_(currentFmt.toString()).m_6270_(zeroWidth));
        }
        return display;
    }
}
