package org.moddingx.libx.datagen.provider.sandbox;

import net.minecraft.core.BlockPos;
import net.minecraft.core.Holder;
import net.minecraft.core.Vec3i;
import net.minecraft.core.registries.Registries;
import net.minecraft.data.worldgen.placement.PlacementUtils;
import net.minecraft.tags.TagKey;
import net.minecraft.util.valueproviders.ConstantInt;
import net.minecraft.util.valueproviders.IntProvider;
import net.minecraft.util.valueproviders.UniformInt;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.levelgen.Heightmap;
import net.minecraft.world.level.levelgen.VerticalAnchor;
import net.minecraft.world.level.levelgen.blockpredicates.BlockPredicate;
import net.minecraft.world.level.levelgen.carver.CarverConfiguration;
import net.minecraft.world.level.levelgen.carver.ConfiguredWorldCarver;
import net.minecraft.world.level.levelgen.carver.WorldCarver;
import net.minecraft.world.level.levelgen.feature.ConfiguredFeature;
import net.minecraft.world.level.levelgen.feature.Feature;
import net.minecraft.world.level.levelgen.feature.configurations.FeatureConfiguration;
import net.minecraft.world.level.levelgen.feature.configurations.NoneFeatureConfiguration;
import net.minecraft.world.level.levelgen.placement.*;
import net.minecraft.world.level.material.Fluids;
import org.moddingx.libx.datagen.DatagenContext;
import org.moddingx.libx.datagen.DatagenStage;
import org.moddingx.libx.datagen.provider.RegistryProviderBase;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

/**
 * SandBox provider for {@link ConfiguredFeature configured} and {@link PlacedFeature placed features}.
 *
 * This provider must run in the {@link DatagenStage#REGISTRY_SETUP registry setup} stage.
 */
public abstract class FeatureProviderBase extends RegistryProviderBase {

    protected FeatureProviderBase(DatagenContext ctx) {
        super(ctx, DatagenStage.REGISTRY_SETUP);
    }

    @Override
    public final String getName() {
        return this.mod.modid + " features";
    }
    
    /**
     * Makes a new {@link ConfiguredWorldCarver configured feature} without configuration.
     *
     * This method returns an {@link Holder.Reference.Type#INTRUSIVE intrusive holder} that must be properly
     * added the registry. {@link RegistryProviderBase} does this automatically if the result is stored in a
     * {@code public}, non-{@code static} field inside the provider.
     */
    public Holder<ConfiguredFeature<?, ?>> feature(Feature<NoneFeatureConfiguration> feature) {
        return this.feature(feature, NoneFeatureConfiguration.f_67816_);
    }

    /**
     * Makes a new {@link ConfiguredWorldCarver configured feature}.
     *
     * This method returns an {@link Holder.Reference.Type#INTRUSIVE intrusive holder} that must be properly
     * added the registry. {@link RegistryProviderBase} does this automatically if the result is stored in a
     * {@code public}, non-{@code static} field inside the provider.
     */
    public <C extends FeatureConfiguration> Holder<ConfiguredFeature<?, ?>> feature(Feature<C> feature, C config) {
        return this.registries.writableRegistry(Registries.f_256911_).m_203693_(new ConfiguredFeature<>(feature, config));
    }

    /**
     * Makes a new {@link ConfiguredWorldCarver configured carver}.
     *
     * This method returns an {@link Holder.Reference.Type#INTRUSIVE intrusive holder} that must be properly
     * added the registry. {@link RegistryProviderBase} does this automatically if the result is stored in a
     * {@code public}, non-{@code static} field inside the provider.
     */
    public <C extends CarverConfiguration> Holder<ConfiguredWorldCarver<?>> carver(WorldCarver<C> carver, C config) {
        return this.registries.writableRegistry(Registries.f_257003_).m_203693_(new ConfiguredWorldCarver<>(carver, config));
    }

    /**
     * Returns a new builder for a {@link PlacedFeature}.
     */
    public PlacementBuilder placement(Holder<ConfiguredFeature<?, ?>> feature) {
        return new PlacementBuilder(feature);
    }
    
    /**
     * Returns a new builder for placement modifiers.
     */
    public ModifierBuilder modifiers() {
        return new ModifierBuilder();
    }

    public abstract static class AnyPlacementBuilder<T> {
        
        protected final List<PlacementModifier> modifiers;

        private AnyPlacementBuilder() {
            this.modifiers = new ArrayList<>();
        }
        
        public AnyPlacementBuilder<T> count(int count) {
            if (count > 1) {
                return this.count(ConstantInt.m_146483_(count));
            } else {
                return this;
            }
        }
        
        public AnyPlacementBuilder<T> count(int min, int max) {
            if (min == max) {
                return this.count(min);
            } else {
                return this.count(UniformInt.m_146622_(min, max));
            }
        }
        
        public AnyPlacementBuilder<T> count(IntProvider count) {
            return this.add(CountPlacement.m_191630_(count));
        }
        
        public AnyPlacementBuilder<T> countExtra(int base, float chance, int extra) {
            return this.add(PlacementUtils.m_195364_(base, chance, extra));
        }

        public AnyPlacementBuilder<T> rarity(int avgOnceEveryChunk) {
            return this.add(RarityFilter.m_191900_(avgOnceEveryChunk));
        }
        
        public AnyPlacementBuilder<T> noiseCount(int noiseToCount, double factor, double offset) {
            return this.add(NoiseBasedCountPlacement.m_191731_(noiseToCount, factor, offset));
        }

        public AnyPlacementBuilder<T> noiseThresholdCount(double noiseLevel, int above, int below) {
            return this.add(NoiseThresholdCountPlacement.m_191756_(noiseLevel, above, below));
        }
        
        public AnyPlacementBuilder<T> spread() {
            return this.add(InSquarePlacement.m_191715_());
        }
        
        public AnyPlacementBuilder<T> height(VerticalAnchor bottom, VerticalAnchor top) {
            return this.add(HeightRangePlacement.m_191680_(bottom, top));
        }
        
        public AnyPlacementBuilder<T> heightTriangle(VerticalAnchor bottom, VerticalAnchor top) {
            return this.add(HeightRangePlacement.m_191692_(bottom, top));
        }
        
        public AnyPlacementBuilder<T> heightmap(Heightmap.Types type) {
            return this.add(HeightmapPlacement.m_191702_(type));
        }
        
        public AnyPlacementBuilder<T> biomeFilter() {
            return this.add(BiomeFilter.m_191561_());
        }

        public AnyPlacementBuilder<T> validGround(TagKey<Block> tag) {
            return this.add(BlockPredicateFilter.m_191576_(BlockPredicate.m_190404_(
                    BlockPredicate.m_224768_(new Vec3i(0, -1, 0), tag),
                    BlockPredicate.m_224782_(Fluids.f_76191_)
            )));
        }
        
        public AnyPlacementBuilder<T> validGround(Block block) {
            return this.validGround(block.m_49966_());
        }
        
        public AnyPlacementBuilder<T> validGround(BlockState state) {
            return this.add(BlockPredicateFilter.m_191576_(BlockPredicate.m_190404_(
                    BlockPredicate.m_190399_(state, BlockPos.f_121853_),
                    BlockPredicate.m_224782_(Fluids.f_76191_)
            )));
        }
        
        public AnyPlacementBuilder<T> waterDepth(int maxDepth) {
            return this.add(SurfaceWaterDepthFilter.m_191950_(maxDepth));
        }
        
        public AnyPlacementBuilder<T> inAir() {
            return this.add(PlacementUtils.m_206517_());
        }
        
        public AnyPlacementBuilder<T> add(PlacementModifiers modifiers) {
            return this.addAll(modifiers.modifiers);
        }
        
        public AnyPlacementBuilder<T> add(PlacementModifier... modifiers) {
            this.modifiers.addAll(Arrays.asList(modifiers));
            return this;
        }
        
        public AnyPlacementBuilder<T> addAll(Collection<PlacementModifier> modifiers) {
            this.modifiers.addAll(modifiers);
            return this;
        }
        
        public abstract T build();
    }
    
    public class PlacementBuilder extends AnyPlacementBuilder<Holder<PlacedFeature>> {

        private final Holder<ConfiguredFeature<?, ?>> feature;

        private PlacementBuilder(Holder<ConfiguredFeature<?, ?>> feature) {
            this.feature = feature;
        }

        /**
         * Builds the {@link PlacedFeature}.
         *
         * This method returns an {@link Holder.Reference.Type#INTRUSIVE intrusive holder} that must be properly
         * added the registry. {@link RegistryProviderBase} does this automatically if the result is stored in a
         * {@code public}, non-{@code static} field inside the provider.
         */
        @Override
        public Holder<PlacedFeature> build() {
            return FeatureProviderBase.this.registries.writableRegistry(Registries.f_256988_).m_203693_(new PlacedFeature(this.feature, List.copyOf(this.modifiers)));
        }
    }
    
    public static class ModifierBuilder extends AnyPlacementBuilder<PlacementModifiers> {
        
        private ModifierBuilder() {
            
        }

        @Override
        public PlacementModifiers build() {
            return new PlacementModifiers(List.copyOf(this.modifiers));
        }
    }
    
    public static class PlacementModifiers {
        
        private final List<PlacementModifier> modifiers;

        public PlacementModifiers(List<PlacementModifier> modifiers) {
            this.modifiers = List.copyOf(modifiers);
        }
    }
}
