package org.moddingx.libx.datagen.provider.sandbox;

import net.minecraft.core.Holder;
import net.minecraft.core.registries.Registries;
import net.minecraft.tags.TagKey;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.levelgen.structure.templatesystem.*;
import net.minecraft.world.level.levelgen.structure.templatesystem.rule.blockentity.Passthrough;
import net.minecraft.world.level.levelgen.structure.templatesystem.rule.blockentity.RuleBlockEntityModifier;
import org.moddingx.libx.datagen.DatagenContext;
import org.moddingx.libx.datagen.DatagenStage;
import org.moddingx.libx.datagen.provider.RegistryProviderBase;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

/**
 * SandBox provider for {@link StructureProcessorList structure processors}.
 *
 * This provider must run in the {@link DatagenStage#REGISTRY_SETUP registry setup} stage.
 */
public abstract class StructureProcessorProviderBase extends RegistryProviderBase {

    protected StructureProcessorProviderBase(DatagenContext ctx) {
        super(ctx, DatagenStage.REGISTRY_SETUP);
    }

    @Override
    public final String getName() {
        return this.mod.modid + " structure processors";
    }

    /**
     * Returns a new builder for a processor list.
     */
    public ProcessorListBuilder processor() {
        return new ProcessorListBuilder();
    }
    
    /**
     * Returns a new builder for a processor rule.
     */
    public static ProcessorRuleBuilder rule(Block block) {
        return rule(block.m_49966_());
    }

    /**
     * Returns a new builder for a processor rule.
     */
    public static ProcessorRuleBuilder rule(Block block, RuleBlockEntityModifier modifier) {
        return rule(block.m_49966_(), modifier);
    }

    /**
     * Returns a new builder for a processor rule.
     */
    public static ProcessorRuleBuilder rule(BlockState state) {
        return rule(state, Passthrough.f_276645_);
    }

    /**
     * Returns a new builder for a processor rule.
     */
    public static ProcessorRuleBuilder rule(BlockState state, RuleBlockEntityModifier modifier) {
        return new ProcessorRuleBuilder(state, modifier);
    }
    
    public class ProcessorListBuilder {
        
        private final List<StructureProcessor> processors;
        private final List<ProcessorRule> rules;

        private ProcessorListBuilder() {
            this.processors = new ArrayList<>();
            this.rules = new ArrayList<>();
        }
        
        public ProcessorListBuilder jigsaw() {
            return this.add(JigsawReplacementProcessor.f_74122_);
        }
        
        public ProcessorListBuilder keep(TagKey<Block> protectedBlocks) {
            return this.add(new ProtectedBlockProcessor(protectedBlocks));
        }
        
        public ProcessorListBuilder ignore(List<Block> ignoredBlocks) {
            return this.add(new BlockIgnoreProcessor(List.copyOf(ignoredBlocks)));
        }
        
        public ProcessorListBuilder age(float mossProbability) {
            return this.add(new BlockAgeProcessor(mossProbability));
        }
        
        public ProcessorListBuilder addRule(ProcessorRule... rules) {
            this.rules.addAll(Arrays.asList(rules));
            return this;
        }
        
        public ProcessorListBuilder addRules(Collection<ProcessorRule> rules) {
            this.rules.addAll(rules);
            return this;
        }
        
        public ProcessorListBuilder add(StructureProcessor... processors) {
            this.rulesToProcessor();
            this.processors.addAll(Arrays.asList(processors));
            return this;
        }
        
        public ProcessorListBuilder addAll(Collection<StructureProcessor> processors) {
            this.rulesToProcessor();
            this.processors.addAll(processors);
            return this;
        }
        
        private void rulesToProcessor() {
            if (!this.rules.isEmpty()) {
                this.processors.add(new RuleProcessor(List.copyOf(this.rules)));
                this.rules.clear();
            }
        }

        /**
         * Builds the {@link StructureProcessorList}.
         *
         * This method returns an {@link Holder.Reference.Type#INTRUSIVE intrusive holder} that must be properly
         * added the registry. {@link RegistryProviderBase} does this automatically if the result is stored in a
         * {@code public}, non-{@code static} field inside the provider.
         */
        public Holder<StructureProcessorList> build() {
            this.rulesToProcessor();
            if (this.processors.isEmpty()) this.add(NopProcessor.f_74175_);
            return StructureProcessorProviderBase.this.registries.writableRegistry(Registries.f_257011_).m_203693_(new StructureProcessorList(List.copyOf(this.processors)));
        }
    }

    public static class ProcessorRuleBuilder {

        private final BlockState output;
        private final RuleBlockEntityModifier modifier;
        private RuleTest templateState;
        private RuleTest worldState;
        private PosRuleTest location;

        private ProcessorRuleBuilder(BlockState output, RuleBlockEntityModifier modifier) {
            this.output = output;
            this.modifier = modifier;
            this.templateState = AlwaysTrueTest.f_73954_;
            this.worldState = AlwaysTrueTest.f_73954_;
            this.location = PosAlwaysTrueTest.f_74188_;
        }

        public ProcessorRuleBuilder templateStateTest(RuleTest templateState) {
            this.templateState = templateState;
            return this;
        }

        public ProcessorRuleBuilder worldStateTest(RuleTest worldState) {
            this.worldState = worldState;
            return this;
        }

        public ProcessorRuleBuilder locationTest(PosRuleTest location) {
            this.location = location;
            return this;
        }

        public ProcessorRule build() {
            return new ProcessorRule(this.templateState, this.worldState, this.location, this.output, this.modifier);
        }
    }
}
