package org.moddingx.libx.menu;

import com.google.common.collect.ImmutableList;
import net.minecraft.network.chat.Component;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.MenuProvider;
import net.minecraft.world.entity.player.Inventory;
import net.minecraft.world.entity.player.Player;
import net.minecraft.world.inventory.AbstractContainerMenu;
import net.minecraft.world.inventory.MenuType;
import net.minecraft.world.inventory.Slot;
import net.minecraft.world.item.ItemStack;
import net.minecraftforge.common.extensions.IForgeMenuType;
import net.minecraftforge.items.IItemHandlerModifiable;
import net.minecraftforge.items.ItemStackHandler;
import net.minecraftforge.items.SlotItemHandler;
import net.minecraftforge.network.NetworkHooks;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.moddingx.libx.LibX;
import org.moddingx.libx.impl.menu.GenericContainerSlotValidationWrapper;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiPredicate;

/**
 * A common {@link AbstractContainerMenu menu} with a variable amount of slots. You should not use this with
 * more than 154 slots.
 * 
 * To show a menu to a {@link Player player}, call
 * {@link #open(ServerPlayer, IItemHandlerModifiable, Component, ResourceLocation) open} on the logical server.
 * 
 * As there's no way to synchronise the item validator method from the {@link IItemHandlerModifiable modifiable item handler},
 * you should register the validator during setup. The slot validation method in your item handler will be ignored by this.
 */
public class GenericMenu extends MenuBase {

    private static final ResourceLocation EMPTY_VALIDATOR = LibX.getInstance().resource("nothing");
    private static final Map<ResourceLocation, BiPredicate<Integer, ItemStack>> validators = new HashMap<>(Map.of(
            EMPTY_VALIDATOR, (slot, stack) -> true
    ));

    public static final MenuType<GenericMenu> TYPE = IForgeMenuType.create((id, playerInv, buffer) -> {
        int size = buffer.m_130242_();
        ResourceLocation validatorId = buffer.m_130281_();
        BiPredicate<Integer, ItemStack> validator;
        if (validators.containsKey(validatorId)) {
            validator = validators.get(validatorId);
        } else {
            LibX.logger.warn("Received invalid validator for generic container. Validator: " + validatorId);
            validator = validators.get(EMPTY_VALIDATOR);
        }
        int[] slotLimits = new int[size];
        for (int i = 0; i < size; i++) {
            slotLimits[i] = buffer.m_130242_();
        }
        IItemHandlerModifiable handler = new GenericContainerSlotValidationWrapper(new ItemStackHandler(size), validator, slotLimits);
        return new GenericMenu(id, handler, playerInv);
    });

    public final int width;
    public final int height;
    public final int invX;
    public final int invY;
    public final List<Pair<Integer, Integer>> slotList;

    private GenericMenu(int id, IItemHandlerModifiable handler, Inventory playerContainer) {
        super(TYPE, id, playerContainer);
        Triple<Pair<Integer, Integer>, Pair<Integer, Integer>, List<Pair<Integer, Integer>>> layout = layoutSlots(handler.getSlots());
        this.width = layout.getLeft().getLeft();
        this.height = layout.getLeft().getRight();
        this.invX = layout.getMiddle().getLeft();
        this.invY = layout.getMiddle().getRight();
        this.slotList = layout.getRight();
        for (int i = 0; i < this.slotList.size(); i++) {
            this.m_38897_(new SlotItemHandler(handler, i, this.slotList.get(i).getLeft(), this.slotList.get(i).getRight()));
        }
        this.layoutPlayerInventorySlots(layout.getMiddle().getLeft(), layout.getMiddle().getRight());
    }

    @Override
    public boolean m_6875_(@Nonnull Player player) {
        return true;
    }

    @Nonnull
    @Override
    public ItemStack m_7648_(@Nonnull Player player, int index) {
        ItemStack itemstack = ItemStack.f_41583_;
        Slot slot = this.f_38839_.get(index);
        if (slot.m_6657_()) {
            ItemStack stack = slot.m_7993_();
            itemstack = stack.m_41777_();

            final int inventorySize = this.slotList.size();
            final int playerInventoryEnd = inventorySize + 27;
            final int playerHotBarEnd = playerInventoryEnd + 9;

            if (index >= inventorySize) {
                if (!this.m_38903_(stack, 0, inventorySize, false)) {
                    return ItemStack.f_41583_;
                } else if (index < playerInventoryEnd) {
                    if (!this.m_38903_(stack, playerInventoryEnd, playerHotBarEnd, false)) {
                        return ItemStack.f_41583_;
                    }
                } else if (index < playerHotBarEnd && !this.m_38903_(stack, inventorySize, playerInventoryEnd, false)) {
                    return ItemStack.f_41583_;
                }
            } else if (!this.m_38903_(stack, inventorySize, playerHotBarEnd, false)) {
                return ItemStack.f_41583_;
            }
            if (stack.m_41619_()) {
                slot.m_5852_(ItemStack.f_41583_);
            } else {
                slot.m_6654_();
            }
            if (stack.m_41613_() == itemstack.m_41613_()) {
                return ItemStack.f_41583_;
            }
            slot.m_142406_(player, stack);
        }
        return itemstack;
    }

    /**
     * Opens a menu for a {@link Player player}.
     *
     * @param player      The player that should see the menu.
     * @param inventory   The inventory of the menu. The slot amount of this determines how big the container is.
     *                    This should not have more than 154 slots.
     * @param name        The name of the menu.
     * @param validatorId The id of the slot validator registered with {@link #registerSlotValidator(ResourceLocation, BiPredicate) registerSlotValidator}.
     *                    {@code null} disables slot validation. This will override the item handlers slot validation, so null
     *                    means no slot validation even if the item handler has that feature.
     */
    public static void open(ServerPlayer player, IItemHandlerModifiable inventory, Component name, @Nullable ResourceLocation validatorId) {
        MenuProvider provider = new MenuProvider() {

            @Nonnull
            @Override
            public Component m_5446_() {
                return name;
            }

            @Override
            public AbstractContainerMenu m_7208_(int containerId, @Nonnull Inventory inv, @Nonnull Player player) {
                BiPredicate<Integer, ItemStack> validator;
                if (validators.containsKey(validatorId == null ? EMPTY_VALIDATOR : validatorId)) {
                    validator = validators.get(validatorId);
                } else {
                    LibX.logger.warn("Generic container created with invalid validator. Validator ID: " + validatorId);
                    validator = validators.get(EMPTY_VALIDATOR);
                }
                return new GenericMenu(containerId, new GenericContainerSlotValidationWrapper(inventory, validator, null), inv);
            }
        };
        NetworkHooks.openScreen(player, provider, buffer -> {
            buffer.m_130130_(inventory.getSlots());
            buffer.m_130085_(validatorId == null ? EMPTY_VALIDATOR : validatorId);
            for (int i = 0; i < inventory.getSlots(); i++) {
                buffer.m_130130_(inventory.getSlotLimit(i));
            }
        });
    }

    /**
     * Registers a slot validator. This is required as the item handler can not be synced to the client,
     * so the slot validation method of the item handler can not be used. This should be called during setup.
     */
    public static void registerSlotValidator(ResourceLocation validatorId, BiPredicate<Integer, ItemStack> validator) {
        synchronized (validators) {
            if (validators.containsKey(validatorId)) {
                throw new IllegalStateException("Slot validator for generic container registered: " + validatorId);
            }
            validators.put(validatorId, validator);
        }
    }

    private static Triple<Pair<Integer, Integer>, Pair<Integer, Integer>, List<Pair<Integer, Integer>>> layoutSlots(int size) {
        // We try some special cases here for the best possible results.
        // If nothing works we just put them in a rectangle
        if (size < 9) {
            return layoutRectangle(size, 1, size);
        } else if (size % 9 == 0 && size <= 9 * 8) {
            return layoutRectangle(9, size / 9, size);
        } else if (size % 11 == 0 && size <= 11 * 8) {
            return layoutRectangle(11, size / 11, size);
        } else if (size % 12 == 0 && size <= 12 * 8) {
            return layoutRectangle(12, size / 12, size);
        } else if (size % 8 == 0 && size <= 8 * 8) {
            return layoutRectangle(8, size / 8, size);
        } else if (size % 13 == 0 && size <= 13 * 8) {
            return layoutRectangle(13, size / 13, size);
        } else if (size % 14 == 0 && size <= 14 * 8) {
            return layoutRectangle(14, size / 14, size);
        } else if (size <= 9 * 8) {
            return layoutRectangle(9, size % 9 == 0 ? size / 9 : (size / 9) + 1, size);
        } else if (size <= 11 * 8) {
            return layoutRectangle(11, size % 11 == 0 ? size / 11 : (size / 11) + 1, size);
        } else if (size <= 12 * 8) {
            return layoutRectangle(12, size % 12 == 0 ? size / 12 : (size / 12) + 1, size);
        } else if (size <= 13 * 8) {
            return layoutRectangle(13, size % 13 == 0 ? size / 13 : (size / 13) + 1, size);
        } else {
            return layoutRectangle(14, size % 14 == 0 ? size / 14 : (size / 14) + 1, size);
        }
    }

    private static Triple<Pair<Integer, Integer>, Pair<Integer, Integer>, List<Pair<Integer, Integer>>> layoutRectangle(int width, int height, int maxSize) {
        int invX;
        int paddingX;
        if (width < 9) {
            invX = 0;
            paddingX = (9 - width) * 9;
        } else {
            invX = (width - 9) * 9;
            paddingX = 0;
        }
        ImmutableList.Builder<Pair<Integer, Integer>> builder = ImmutableList.builder();
        for (int y = 0; y < height; y++) {
            for (int x = 0; x < width; x++) {
                if (((y * width) + x) < maxSize)
                    builder.add(Pair.of(7 + paddingX + (18 * x) + 1, 17 + (18 * y) + 1));
            }
        }
        return Triple.of(
                Pair.of(
                        Math.max((2 * (7 + invX)) + (9 * 18), (2 * (7 + paddingX)) + (width * 18)),
                        17 + (18 * height) + 14 + 83
                ),
                Pair.of(7 + invX + 1, 17 + (height * 18) + 14 + 1),
                builder.build()
        );
    }
}
