package org.moddingx.libx.menu;

import com.mojang.datafixers.util.Function4;
import net.minecraft.world.entity.player.Inventory;
import net.minecraft.world.inventory.AbstractContainerMenu;
import net.minecraft.world.inventory.MenuType;
import net.minecraft.world.inventory.Slot;
import net.minecraft.world.item.ItemStack;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.SlotItemHandler;
import net.minecraftforge.items.wrapper.InvWrapper;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * A base class for {@link AbstractContainerMenu menus}. Provides some utilities that are useful for any type
 * of menu. When using this it's important to register the player inventory slots through
 * {@link #layoutPlayerInventorySlots(int, int)} and after all the other slots.
 */
public abstract class MenuBase extends AbstractContainerMenu {
    
    public final IItemHandler playerInventory;
    
    protected MenuBase(@Nullable MenuType<?> type, int id, Inventory inventory) {
        super(type, id);
        this.playerInventory = new InvWrapper(inventory);
    }

    /**
     * Places the player inventory slots into the container.
     *
     * @param leftCol The x coordinate of the top left slot
     * @param topRow  The y coordinate of the top left slot
     */
    protected void layoutPlayerInventorySlots(int leftCol, int topRow) {
        this.addSlotBox(this.playerInventory, 9, leftCol, topRow, 9, 18, 3, 18);
        topRow += 58;
        this.addSlotRange(this.playerInventory, 0, leftCol, topRow, 9, 18);
    }

    /**
     * Adds a box of slots to the container
     *
     * @param handler   The inventory of the slot
     * @param index     The index of the first slot
     * @param x         The x coordinate of the top left slot
     * @param y         The y coordinate of the top left slot
     * @param horAmount The amount of slots in horizontal direction
     * @param dx        The space between two slots in horizontal direction. Should not be less than 16 or
     *                  you create overlapping slots. Most of the time this is 18
     * @param verAmount The amount of slots in vertical direction
     * @param dy        The space between two slots in vertical direction. Should not be less than 16 or
     *                  you create overlapping slots. Most of the time this is 18
     * @return The next index to be used to create a slot
     */
    protected int addSlotBox(IItemHandler handler, int index, int x, int y, int horAmount, int dx, int verAmount, int dy) {
        return this.addSlotBox(handler, index, x, y, horAmount, dx, verAmount, dy, SlotItemHandler::new);
    }

    /**
     * Adds a row of slots to the container
     *
     * @param handler The inventory of the slot
     * @param index   The index of the first slot
     * @param x       The x coordinate of the top left slot
     * @param y       The y coordinate of the top left slot
     * @param amount  The amount of slots
     * @param dx      The space between two slots. Should not be less than 16 or
     *                you create overlapping slots. Most of the time this is 18
     * @return The next index to be used to create a slot
     */
    protected int addSlotRange(IItemHandler handler, int index, int x, int y, int amount, int dx) {
        return this.addSlotRange(handler, index, x, y, amount, dx, SlotItemHandler::new);
    }

    /**
     * Adds a box of slots to the container
     *
     * @param handler     The inventory of the slot
     * @param index       The index of the first slot
     * @param x           The x coordinate of the top left slot
     * @param y           The y coordinate of the top left slot
     * @param horAmount   The amount of slots in horizontal direction
     * @param dx          The space between two slots in horizontal direction. Should not be less than 16 or
     *                    you create overlapping slots. Most of the time this is 18
     * @param verAmount   The amount of slots in vertical direction
     * @param dy          The space between two slots in vertical direction. Should not be less than 16 or
     *                    you create overlapping slots. Most of the time this is 18
     * @param slotFactory A factory to create a slot. This could be {@code SlotItemHandler::new}
     *                    or {@code SlotOutputOnly::new} for output slots.
     * @return The next index to be used to create a slot
     */
    protected int addSlotBox(IItemHandler handler, int index, int x, int y, int horAmount, int dx, int verAmount, int dy, Function4<IItemHandler, Integer, Integer, Integer, Slot> slotFactory) {
        for (int j = 0; j < verAmount; j++) {
            index = this.addSlotRange(handler, index, x, y, horAmount, dx, slotFactory);
            y += dy;
        }
        return index;
    }

    /**
     * Adds a row of slots to the container
     *
     * @param handler     The inventory of the slot
     * @param index       The index of the first slot
     * @param x           The x coordinate of the top left slot
     * @param y           The y coordinate of the top left slot
     * @param amount      The amount of slots
     * @param dx          The space between two slots. Should not be less than 16 or
     *                    you create overlapping slots. Most of the time this is 18
     * @param slotFactory A factory to create a slot. This could be {@code SlotItemHandler::new}
     *                    or {@code SlotOutputOnly::new} for output slots.
     * @return The next index to be used to create a slot
     */
    protected int addSlotRange(IItemHandler handler, int index, int x, int y, int amount, int dx, Function4<IItemHandler, Integer, Integer, Integer, Slot> slotFactory) {
        for (int i = 0; i < amount; i++) {
            this.m_38897_(slotFactory.apply(handler, index, x, y));
            x += dx;
            index++;
        }
        return index;
    }

    // As opposed to the super method, this checks for Slot#mayPlace(ItemStack)
    @Override
    protected boolean m_38903_(@Nonnull ItemStack stack, int startIndex, int endIndex, boolean reverseDirection) {
        boolean success = false;
        
        if (stack.m_41753_()) {
            int idx = reverseDirection ? endIndex - 1 : startIndex;
            while(!stack.m_41619_()) {
                if (reverseDirection ? idx < startIndex : idx >= endIndex) {
                    break;
                }

                Slot slot = this.f_38839_.get(idx);
                ItemStack content = slot.m_7993_();
                if (!content.m_41619_() && ItemStack.m_150942_(stack, content) && slot.m_5857_(stack)) {
                    int totalCount = content.m_41613_() + stack.m_41613_();
                    int maxSize = Math.min(slot.m_6641_(), stack.m_41741_());
                    if (totalCount <= maxSize) {
                        stack.m_41764_(0);
                        content.m_41764_(totalCount);
                        slot.m_6654_();
                        success = true;
                    } else if (content.m_41613_() < maxSize) {
                        stack.m_41774_(maxSize - content.m_41613_());
                        content.m_41764_(maxSize);
                        slot.m_6654_();
                        success = true;
                    }
                }

                idx += (reverseDirection ? -1 : 1);
            }
        }

        if (!stack.m_41619_()) {
            int idx = reverseDirection ? endIndex - 1 : startIndex;
            while(true) {
                if (reverseDirection ? idx < startIndex : idx >= endIndex) {
                    break;
                }

                Slot slot = this.f_38839_.get(idx);
                ItemStack content = slot.m_7993_();
                if (content.m_41619_() && slot.m_5857_(stack)) {
                    if (stack.m_41613_() > slot.m_6641_()) {
                        slot.m_5852_(stack.m_41620_(slot.m_6641_()));
                    } else {
                        slot.m_5852_(stack.m_41620_(stack.m_41613_()));
                    }
                    slot.m_6654_();
                    success = true;
                    break;
                }

                idx += (reverseDirection ? -1 : 1);
            }
        }

        return success;
    }
}
