package org.moddingx.libx.util.game;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import net.minecraft.ChatFormatting;
import net.minecraft.network.chat.*;
import net.minecraft.util.FormattedCharSequence;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Utilities for {@link Component text components}.
 */
public class ComponentUtil {

    private static final HoverEvent HOVER_COPY = new HoverEvent(HoverEvent.Action.f_130831_, Component.m_237115_("libx.misc.copy"));
    
    /**
     * Gets a {@link Component text component} as a string formatted with ANSI escape codes to
     * be printed on the console.
     */
    public static String getConsoleString(Component tc) {
        StringBuilder sb = new StringBuilder();
        tc.m_7451_((style, string) -> {
            reset(sb);
            formattingCodes(sb, style);
            sb.append(string);
            return Optional.empty();
        }, Style.f_131099_);
        reset(sb);
        return sb.toString();
    }
    
    private static void formattingCodes(StringBuilder sb, Style style) {
        if (style.m_131135_() != null) {
            int color = style.m_131135_().f_131257_;
            sb.append("\u001B[38;2;").append((color >> 16) & 0xFF).append(";").append((color >> 8) & 0xFF).append(";").append(color & 0xFF).append("m");
        }
        if (style.f_131102_ != null) {
            if (style.f_131102_) {
                sb.append("\u001B[1m");
            } else {
                sb.append("\u001B[22m");
            }
        }
        if (style.f_131103_ != null) {
            if (style.f_131103_) {
                sb.append("\u001B[3m");
            } else {
                sb.append("\u001B[23m");
            }
        }
        if (style.f_131104_ != null) {
            if (style.f_131104_) {
                sb.append("\u001B[4m");
            } else {
                sb.append("\u001B[24m");
            }
        }
        if (style.f_131105_ != null) {
            if (style.f_131105_) {
                sb.append("\u001B[9m");
            } else {
                sb.append("\u001B[29m");
            }
        }
        if (style.f_131106_ != null) {
            if (style.f_131106_) {
                sb.append("\u001B[8m");
            } else {
                sb.append("\u001B[28m");
            }
        }
    }
    
    private static void reset(StringBuilder sb) {
        sb.append("\u001B[0m");
    }

    /**
     * Turns a {@link JsonElement} to a {@link Component} with syntax highlighting that can be used for display.
     */
    public static Component toPrettyComponent(JsonElement json) {
        if (json.isJsonNull()) {
            return Component.m_237113_("null").m_130940_(ChatFormatting.RED);
        } else if (json instanceof JsonPrimitive primitive) {
            if (primitive.isString()) {
                return Component.m_237113_(primitive.toString()).m_130940_(ChatFormatting.GREEN);
            } else {
                return Component.m_237113_(primitive.toString()).m_130940_(ChatFormatting.GOLD);
            }
        } else if (json instanceof JsonArray array) {
            MutableComponent tc = Component.m_237113_("[");
            boolean first = true;
            for (JsonElement element : array) {
                if (first) {
                    first = false;
                } else {
                    tc.m_7220_(Component.m_237113_(", "));
                }
                tc = tc.m_7220_(toPrettyComponent(element));
            }
            tc = tc.m_7220_(Component.m_237113_("]"));
            return tc;
        } else if (json instanceof JsonObject object) {
            MutableComponent tc = Component.m_237113_("{");
            boolean first = true;
            for (Map.Entry<String, JsonElement> entry : object.entrySet()) {
                if (first) {
                    first = false;
                } else {
                    tc.m_7220_(Component.m_237113_(", "));
                }
                tc = tc.m_7220_(Component.m_237113_(new JsonPrimitive(entry.getKey()).toString()).m_130940_(ChatFormatting.AQUA))
                        .m_7220_(Component.m_237113_(": "))
                        .m_7220_(toPrettyComponent(entry.getValue()));
            }
            tc = tc.m_7220_(Component.m_237113_("}"));
            return tc;
        } else {
            throw new IllegalArgumentException("JSON type unknown: " + json.getClass());
        }
    }

    /**
     * Adds a {@link ClickEvent click event} to the given component to copy the given text to clipboard.
     */
    public static Component withCopyAction(Component component, String copyText) {
        return component.m_6881_().m_130948_(Style.f_131099_.m_131142_(new ClickEvent(ClickEvent.Action.COPY_TO_CLIPBOARD, copyText)).m_131144_(HOVER_COPY));
    }
    
    /**
     * Gets a sub-sequence from the given {@link FormattedCharSequence}. The sub sequence will include
     * all characters from {@code start} (inclusive) to the end of the sequence.
     */
    public static FormattedCharSequence subSequence(FormattedCharSequence text, int start) {
        if (start == 0) return text;
        return subSequence(text, start, Integer.MAX_VALUE - 1);
    }
    
    /**
     * Gets a sub-sequence from the given {@link FormattedCharSequence}. The sub sequence will include
     * all characters between {@code start} (inclusive) and {@code end} (exclusive). {@code start} and
     * {@code end} may not be negative but may be greater that the length of the sequence.
     */
    public static FormattedCharSequence subSequence(FormattedCharSequence text, int start, int end) {
        if (start < 0 || end < 0) throw new IllegalArgumentException("Negative bounds");
        if (end <= start) return FormattedCharSequence.f_13691_;
        return sink -> {
            AtomicInteger relOff = new AtomicInteger(0);
            AtomicInteger total = new AtomicInteger(0);
            return text.m_13731_((relPosition, style, codePoint) -> {
                int current = total.getAndIncrement();
                if (current == start) {
                    relOff.set(-relPosition);
                } else if (relPosition == 0) {
                    // Next sequence, reset offset
                    relOff.set(0);
                }
                if (current >= start && current < end) {
                    return sink.m_6411_(relPosition + relOff.get(), style, codePoint);
                } else {
                    return true;
                }
            });
        };
    }
}
